/**************************************************************************/
/*                                                                        */
/* Module:             INTS_INI.C                                         */
/*                                                                        */
/* Function:           Interactive Service Interface                      */
/*                     -- Initialization routines                         */
/*                        1.  USHORT  BUSAPIENTRY INTSRV_FP()             */
/*                        2.  USHORT  BUSAPIENTRY INTSRV_CL(...)          */
/*                        3.  USHORT  BUSAPIENTRY INTSRV_PFE(...)         */
/*                        4.  USHORT  BUSAPIENTRY INTSRV_ENV(...)         */
/*                        5.  USHORT  BUSAPIENTRY INTSRV_HLPX(...)        */
/*                        6.  MRESULT BUSAPIENTRY INTSRV_GDA(...)         */
/*                        7.  MRESULT BUSAPIENTRY INTSRV_LAY(...)         */
/*                        8.  LONG INTSRV_RF(...)                         */
/*                                                                        */
/* Author:             W. Reichert, D.3274                                */
/*                                                                        */
/* Date created:       1991-01-14                                         */
/*                                                                        */
/* Classification:     IBM Internal Use Only                              */
/*                                                                        */
/**************************************************************************/
/*                                                                        */
/* Initialization routines:                                               */
/*                                                                        */
/* The initialization routines are possible entry points to INTSRV().     */
/* Application-specific variables may be set and an initialization        */
/* message may be prepared. The arguments of these routines can be        */
/* defined/changed in the header file INTSRV.H.                           */
/*                                                                        */
/* Then INTSRV() is called and its return value is returned to the        */
/* calling program.                                                       */
/*                                                                        */
/* In addition this source module includes the INTS_INI.H file which      */
/* contains the definition of all API test functions.                     */
/* (This means that only this module must be recompiled when adding new   */
/* test functions.)                                                       */
/*                                                                        */
/**************************************************************************/

/**************************************************************************/
/* Include files                                                          */
/**************************************************************************/

#define  INCL_WIN
#include <OS2.H>

#include <stdio.h>
#include <string.h>

#define  INCL_FUNC             /* Include function section in INTS_INI.H  */
#include "INTSRV.H"            /* General INTSRV header file              */
#include "INTS_INI.H"          /* Global function definition file         */

/**************************************************************************/
/* Define directives: Macros for functions with RC of type LONG or ULONG  */
/**************************************************************************/

#define GetValueL(T,N,pV,pL,D) while (GetValue_(T,N,pV,pL,D)) return 1l
#define ShowValueL(T,N,pV,L,D) while (ShowValue_(T,N,pV,L,D)) return 1l

/**************************************************************************/
/* Initialization routines                                                */
/**************************************************************************/

/****** INTSRV_FP() *******************************************************/
                               /**** Calling as function program **********/
USHORT BUSAPIENTRY INTSRV_FP()
                               /* RC: Return code of INTSRV()             */
{
    PSZ pszInitMsg;            /* Initialization message                  */

    pszInitMsg = "Called as FuncProg";

    return INTSRV("FP", pszInitMsg);
}

/****** INTSRV_CL() *******************************************************/
                               /**** Calling as checking logic ************/
USHORT BUSAPIENTRY INTSRV_CL(
    LONG      lVTHandle,       /* i:  VT Handle                           */
    PSZ       pszCommUnit)     /* i:  Communication Unit                  */
                               /* RC: Return code of INTSRV()             */
{
    CHAR achInitMsg[120];      /* Initialization message                  */

    sprintf(achInitMsg, "Called as CheckLogic (CU=\"%.32s\", VTHandle=%li)",
            pszCommUnit, lVTHandle);

    /* Save parmaters as variables in INTSRV pool                         */
    SetValue(TYPE_LONG,  "VTHandle", &lVTHandle,   0L, 0);
    SetValue(TYPE_PSZ,   "CommUnit", &pszCommUnit, 0L, 0);

    return INTSRV("CL", achInitMsg);
}

/****** INTSRV_PFE() ******************************************************/
                               /**** Calling as presentation front end ****/
USHORT BUSAPIENTRY INTSRV_PFE(
    LONG      lVTHandle,       /* i:  VT Handle                           */
    PSZ       pszCommUnit,     /* i:  Communication Unit                  */
    CHAR      chRequest)       /* i:  Request                             */
                               /* RC: Return code of INTSRV()             */
{
    CHAR achInitMsg[120];      /* Initialization message                  */

    sprintf(achInitMsg,
      "Called as PFE (CU=\"%.32s\", VTHandle=%li, Req=\'%c\')",
      pszCommUnit, lVTHandle, chRequest);

    /* Save parameters as variables in INTSRV pool                        */
    SetValue(TYPE_LONG,  "VTHandle", &lVTHandle,   0L, 0);
    SetValue(TYPE_PSZ,   "CommUnit", &pszCommUnit, 0L, 0);

    return INTSRV("PFE", achInitMsg);
}

/****** INTSRV_ENV() ******************************************************/
                               /**** Calling as environment routine *******/
USHORT BUSAPIENTRY INTSRV_ENV(
    PVOID     pEnvAddr,        /* i:  Environment pointer (*obsolete*)    */
    PCHAR     pchData,         /* i:  Data                                */
    ULONG     ulDataLen)       /* i:  Length of data                      */
                               /* RC: Return code of INTSRV()             */
{
    CHAR achInitMsg[120];      /* Initialization message                  */

    sprintf(achInitMsg,
      "Called as ENV (EnvAddr=%p, &EnvData=%p, EnvDataLen=%lu)",
      pEnvAddr, pchData, ulDataLen);

    /* Show data buffer (implicitly saved as INTSRV pool variable)        */
    ShowValue(TYPE_PCHAR, "EnvData", &pchData, ulDataLen, 0);

    return INTSRV("ENV", achInitMsg);
}

/****** INTSRV_HLPX() *****************************************************/
                               /**** Calling as help exit *****************/
USHORT BUSAPIENTRY INTSRV_HLPX(
    LONG      lVTHandle,       /* i:  VT Handle                           */
    PSZ       pszCommUnit,     /* i:  Communication Unit                  */
    PSZ       pszHelpFile,     /* i:  Help file                           */
    PUSHORT   pusHelpID)       /* i/o:Help ID                             */
                               /* RC: Return code of INTSRV()             */
{
    CHAR achInitMsg[120];      /* Initialization message                  */
    USHORT usRC = 0;

    sprintf(achInitMsg, "Called as HelpExit"
            "(CU=\"%.32s\", VTHandle=%li, HelpFile=\"%s\", HelpID=%hu)",
            pszCommUnit, lVTHandle, pszHelpFile, *pusHelpID);

    /* Save parmaters as variables in INTSRV pool                         */
    SetValue(TYPE_LONG,  "VTHandle", &lVTHandle,   0L, 0);
    SetValue(TYPE_PSZ,   "CommUnit", &pszCommUnit, 0L, 0);
    SetValue(TYPE_PSZ,   "HelpFile", &pszHelpFile, 0L, 0);
    SetValue(TYPE_USHORT,"HelpID",   pusHelpID,    0L, 0);
    GetValue(TYPE_USHORT,"HelpID",   pusHelpID,    0,  0);
    GetValue(TYPE_USHORT,"RC",       &usRC,        0,  0);

    return usRC;
}

/****** INTSRV_GDA() ******************************************************/
                               /**** Calling as generic data access *******/
MRESULT BUSAPIENTRY INTSRV_GDA(
    LHANDLE   This,            /* i:  GenOVHa object handle               */
    USHORT    Msg,             /* i:  GenOVHa message                     */
    MPARAM    Mp1,             /* i/o:Input/Output carrier                */
    MPARAM    Mp2)             /* i:  GenOVHa information carrier         */
                               /* RC: 0: Success | 1L: Error              */
{
    CHAR achInitMsg[120];      /* Initialization message                  */

    sprintf(achInitMsg,
            "Called as GDA (This=%lu, Msg=%hu, Mp1=%lu, Mp2=%lu)",
            This, Msg, Mp1, Mp2);

    /* Save parameters as variables in INTSRV pool                        */
    (void) SetValue_(TYPE_LHANDLE, "This",       &This, 0L, 0);
    (void) SetValue_(TYPE_USHORT,  "Msg",        &Msg,  0L, 0);
    (void) SetValue_(TYPE_MPARAM,  "Mp1",        &Mp1,  0L, 0);
    (void) SetValue_(TYPE_MPARAM,  "Mp2",        &Mp2,  0L, 0);

    (void) SetValue_(TYPE_LHANDLE, "hIOCarrier", &Mp1,  0L, 0);

    return (MRESULT) INTSRV("GDA", achInitMsg);
}

/****** INTSRV_LAY() ******************************************************/
                               /**** Calling as layout routine ************/
MRESULT BUSAPIENTRY INTSRV_LAY(
    LHANDLE   Area,            /* i:  GenOVHa area handle                 */
    USHORT    Msg,             /* i:  GenOVHa message                     */
    MPARAM    Mp1,             /* i/o:Input/Output carrier                */
    MPARAM    Mp2)             /* i:  GenOVHa information carrier         */
                               /* RC: 0: Success | 1L: Error              */
{
    CHAR achInitMsg[120];      /* Initialization message                  */

    sprintf(achInitMsg,
            "Called as Layout (Area=%lu, Msg=%hu, Mp1=%lu, Mp2=%lu)",
            Area, Msg, Mp1, Mp2);

    /* Save parameters as variables in INTSRV pool                        */
    (void) SetValue_(TYPE_LHANDLE, "Area",       &Area, 0L, 0);
    (void) SetValue_(TYPE_USHORT,  "Msg",        &Msg,  0L, 0);
    (void) SetValue_(TYPE_MPARAM,  "Mp1",        &Mp1,  0L, 0);
    (void) SetValue_(TYPE_MPARAM,  "Mp2",        &Mp2,  0L, 0);

    return (MRESULT) INTSRV("LAYOUT", achInitMsg);
}

/****** INTSRV_RF() *******************************************************/
                               /**** Calling as remote function ***********/
LONG INTSRV_RF(
    HPCHAR    hpInputBuf,      /* i:  Input Buffer                        */
    ULONG    *pulInputBufLen,  /* i:  Length of input buffer              */
    HPCHAR    hpOutputBuf,     /* o:  Output Buffer                       */
    ULONG    *pulOutputBufLen) /* i/o:Length of output buffer             */
                               /* RC: Return code (LONG!)                 */
{
    LONG lXmitServRC = 0L;

    ShowValueL(TYPE_ACHAR, "InputBuf",     hpInputBuf,  *pulInputBufLen, 0);
    ShowValueL(TYPE_ULONG, "InputBufLen",  pulInputBufLen,           0L, 0);
    ShowValueL(TYPE_ACHAR, "OutputBuf",    hpOutputBuf,*pulOutputBufLen, 0);
    ShowValueL(TYPE_ULONG, "OutputBufLen", pulOutputBufLen,          0L, 0);

    GetValueL(TYPE_ACHAR,  "InputBuf",     hpInputBuf,   pulInputBufLen, 0);
    GetValueL(TYPE_ULONG,  "InputBufLen",  pulInputBufLen,           0L, 0);
    GetValueL(TYPE_ACHAR,  "OutputBuf",    hpOutputBuf, pulOutputBufLen, 0);
    GetValueL(TYPE_ULONG,  "OutputBufLen", pulOutputBufLen,          0L, 0);
    GetValueL(TYPE_LONG,   "lXmitServRC",  &lXmitServRC,             0L, 0);

    return lXmitServRC;
}
